function [ddata,odata,cdata] = colo_sim(num,vsize,varargin)
%This script simulates various outcomes of colocalization analysis using
%random generated data of different densities and clumpiness.
%Synatax:   [ddata,cdata] = colo_sim(50000,[2000 1500 100],'dstep',0.1); 
%Input:     num = the number of randomly generated vertices
%           sdist = the search distance.
%           vsize = the size of the volume [x y z].  The numbers being
%               the max.
%           dstep = the step size for density change.  Default = 10% or .1
%           drange = the range of density, as a percentage of the initial
%                   generated number. Default = 10%-200%
%           cstep = the step size for clumpiness change.  Default = 5% or
%                   0.05
%           crange = the range of clumpiness, as a percentage of the size
%                   of the volume.  Default = 5%-90%
%           't_range' = the distance range you want to map out a three number
%                 value [a b c].  a = min, b = max, c = increment.
%                 Default = [0 15 0.1].  So 150 steps.
%           seed = the initial number of vertices to generate. Default =
%                   5000.
%           forf = "friend or foe" whether the thresh is used to include or
%                   exclude.  1 = include. 0 = exclude
%Output:    ddata & cdata = a structure containing fields:g
%               colo_num = the change in colo as a function of puncta
%                   density or clumpiness.
%               slope = the change in slope as a function of puncta
%                   density or clumpiness.
%               mean = the means of the distances
%               median = the medians of the distances
%               std = the standard deviation of the distances
%               threshold = the distance threshold per point. (not clear
%                           but live with it)

%parse input
[dstep,drange,cstep,crange,t_range,seed,forf,sdist] = parse(num,varargin);

%generate the figures
%density Figures
h1 = figure;    %Density CHange
title('Colocalization as a Function of Density');
ylabel('# of Colocalization Events');
xlabel(['Threshold Distance x',num2str(t_range(3))]);
ax1 = gca;
hold
%slope figure
h2 = figure;    %Density Slope
title('Slope of Change as a Function of Density');
ylabel('Slope')
xlabel(['Threshold Distance x',num2str(t_range(3))]);
ax2 = gca;
hold
%colocolazation Figures
h5 = figure;    %Density CHange
title('Colocalization as a Function of Colocalization');
ylabel('# of Colocalization Events');
xlabel(['Threshold Distance x',num2str(t_range(3))]);
ax5 = gca;
hold
%slope figure
h6 = figure;    %Density Slope
title('Slope of Change as a Function of Colocalization');
ylabel('Slope')
xlabel(['Threshold Distance x',num2str(t_range(3))]);
ax6 = gca;
hold
%Clumpiness Figures
h3 = figure;    %Density CHange
title('Colocalization as a Function of Clumpiness');
ylabel('# of Colocalization Events');
xlabel(['Threshold Distance x',num2str(t_range(3))]);
ax3 = gca;
hold
%slope figure
h4 = figure;    %Density Slope
title('Slope of Change as a Function of Clumpiness');
ylabel('Slope')
xlabel(['Threshold Distance x',num2str(t_range(3))]);
ax4 = gca;
hold

%we are going to keep vert1 the same
vert1 = rand(num,3).*repmat(vsize,num,1);

%density first
currd = drange(1)-dstep;  %the modification on the current density.  Start at the begining
for i = 1:ceil((drange(2)-drange(1))/dstep)   %go through the iterations.
    %prepare density number generation
    tmpd = 1*(currd+dstep*i);
    %lets calculate the density first.
    vert2 = rand(num*tmpd,3).*repmat(vsize,num*tmpd,1);
    %now calculate the metrics
    data_tmp = colo_info('sdist',sdist,'vert1',vert1,'vert2',vert2,'fig',0,'t_range',t_range,'sav',0);  %no figures no saving.
    %now pull the data
    dcolo = data_tmp(1).colo_trend;
    dslope = data_tmp(1).slope;
    dmean = data_tmp(1).mean;
    dmedian = data_tmp(1).median;
    dstd = data_tmp(1).std;
    dthresh = data_tmp(1).thresh_dist;
    %first plot the data
    %prep some plotting information
    dleg{i} = num2str(num*tmpd);
    color = rand(1,3);
    %density first
    figure(h1)
    plot(dcolo(2:end),'color',color);   %Ploting from 1 to end, shifts the cut off, because the thresh starts at 0
    legend(ax1,dleg);
    %slope second
    figure(h2)
    plot(dslope(2:end),'color',color);
    legend(ax2,dleg);
    %now store the data
    ddata(i).colo_trend = dcolo;
    ddata(i).slope = dslope;
    ddata(i).mean = dmean;
    ddata(i).median = dmedian;
    ddata(i).std = dstd;
    ddata(i).thresh_dist = dthresh;
end

%colocalization second
%we are goin to use drange for this.
currd = drange(1)-dstep;  %the modification on the current density.  Start at the begining
for i = 1:ceil((drange(2)-drange(1))/dstep)   %go through the iterations.
    %prepare colo number generation
    tmpd = 1*(currd+dstep*i);
    %lets calculate the colo set first first.
    sign = rand(num,1);
    sign(sign>=0.5) = 1;
    sign(sign<0.5) = -1;
    vert2 = abs(vert1+((rand(num,3)*tmpd).*repmat(sign,1,3)));
    %now calculate the metrics
    data_tmp = colo_info('sdist',sdist,'vert1',vert1,'vert2',vert2,'fig',0,'t_range',t_range,'sav',0);  %no figures no saving.
    %now pull the data
    dcolo = data_tmp(1).colo_trend;
    dslope = data_tmp(1).slope;
    dmean = data_tmp(1).mean;
    dmedian = data_tmp(1).median;
    dstd = data_tmp(1).std;
    dthresh = data_tmp(1).thresh_dist;
    %first plot the data
    %prep some plotting information
    dleg{i} = num2str(tmpd);
    color = rand(1,3);
    %density first
    figure(h5)
    plot(dcolo(2:end),'color',color);   %Ploting from 1 to end, shifts the cut off, because the thresh starts at 0
    legend(ax5,dleg);
    %slope second
    figure(h6)
    plot(dslope(2:end),'color',color);
    legend(ax6,dleg);
    %now store the data
    odata(i).colo_trend = dcolo;
    odata(i).slope = dslope;
    odata(i).mean = dmean;
    odata(i).median = dmedian;
    odata(i).std = dstd;
    odata(i).thresh_dist = dthresh;
end


%clumpiness second
%calculate the first base threshold for clumpiness
%currc = dddist([vsize(1)/2 vsize(2)/2 vsize(3)/2],[vsize(1) vsize(2) vsize(3)]);  %base distance
currc = sdist*2;
currstep = crange(1)-cstep;
j = 0;
while (currstep+cstep*j^2)<crange(2)   %go through the iterations.
    j = j+1;
    tmpc = currc*(currstep+cstep*j^2); %the current threshold
    %lets calculate the varing clumpinesses.
    vert2 = corr_rnd_verts(num,vsize,tmpc,forf);
    %now calculate the metrics
    data_tmp = colo_info('sdist',sdist,'vert1',vert1,'vert2',vert2,'fig',0,'t_range',t_range,'sav',0);
    %now pull the data
    ccolo = data_tmp(1).colo_trend;
    cslope = data_tmp(1).slope;
    cmean = data_tmp(1).mean;
    cmedian = data_tmp(1).median;
    cstd = data_tmp(1).std;
    cthresh = data_tmp(1).thresh_dist;
    %first plot the data
    color = rand(1,3);
    cleg{j} = [num2str(tmpc) '-' num2str(cstep*j^2)];
    %density first
    figure(h3)
    plot(ccolo(2:end),'color',color);
    legend(ax3,cleg);
    %slope second
    figure(h4)
    plot(cslope(2:end),'color',color);
    legend(ax4,cleg);
    %now store the data
    cdata(j).colo_trend = ccolo;
    cdata(j).slope = cslope;
    cdata(j).mean = cmean;
    cdata(j).median = cmedian;
    cdata(j).std = cstd;
    cdata(j).thresh_dist = cthresh;
end
    
%--------------------------------------------------------------------------
%subfunction to parse the inputs.
function [dstep,drange,cstep,crange,t_range,seed,forf,sdist] = parse(num,input)

dstep = 0.1;  %Default Initialized.
drange = [0.1 2];
cstep = 0.01;
crange = [0.01 1];
t_range = [0 15 0.1];
seed = num*0.1;  %Default Generate 10%
forf = 1;   %include
sdist = 10;

%Parse the input
if ~isempty(input)
    for i = 1:2:size(input,2)
        if ischar(input{1,i});
            switch input{1,i}
                case 'dstep'
                    dstep = input{1,i+1};
                case 'sdist'
                    sdist = input{1,i+1};
                case 'drange'
                    drange = input{1,i+1};
                case 'cstep'
                    cstep = input{1,i+1};
                case 'crange'
                    crange = input{1,i+1};
                case 't_range'
                    t_range = input{1,i+1};
                case 'seed'
                    seed = input{1,i+1};
                case 'forf'
                    forf = input{1,i+1};
                otherwise
                    warning(['Your input ',input{1,i},' is not recognized.']);
            end
        end
    end
end